/*-----------------------------------------------------------------------------------------------------------------------------
FileName: readData.c 
Autors: Michel Lavolle, Mathieu Gisselbrecht
Date : Creation - Avril 2003, Revision -  Avril 2006
-----------------------------------------------------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------------------------------------------------------
// PROJECT INCLUDE 
//-----------------------------------------------------------------------------------------------------------------------------
#include "igorTools.h"
#include "XOP_Globals.h"

#include "smallBuffer.h"
#include "particleDetector.h" 
#include "XYDetector.h" 

#include "readData.h"

#ifdef CONNECTED_ON_VME
	#include "PCI-DIO-96-ML.h"
#endif 

//-----------------------------------------------------------------------------------------------------------------------------
// All structures are 2-byte-aligned.
#if TARGET_CPU_PPC
	#pragma options align=mac68k
#endif
#ifdef _WINDOWS_
	#pragma pack(2)
#endif

//-----------------------------------------------------------------------------------------------------------------------------
// MISC #DEFINEs
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Defines

#define INCREASE_STACK 		200000

// #defines for ReadData flags 
//#define OVERWRITE 			1			// /O means overwrite 
#define DOUBLE_PRECISION 	2			// /D means double precision 
#define INTERACTIVE 		4			// /I means interactive -- use open dialog 
#define AUTO_NAME 			8			// /A or /N means autoname wave 
#define PATH 				16			// /P means use symbolic path 
#define QUIET 				32			// /Q means quiet -- no messages in history 
#define FROM_MENU 			64			// Start summoned from menu item 
#define NEW_FORMAT 			128			// data is in new format( CIEL ->2003 )

// #define use for customized error
#define WAVE_DOES_NOT_EXIST			1 + FIRST_XOP_ERR
#define BAD_NUM_WAVES 				2 + FIRST_XOP_ERR			/* number of waves must be >= 1 */

//-----------------------------------------------------------------------------------------------------------------------------
// TYPEDEF AND STRUCTURE 
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Local Structures 

typedef struct LoadParams {
	long flags;					// Flag bits are defined in ReadData.h.
	int mode;
	int	add;				// set by /A flag
	int acquire;
	int calibrate;
	
	long deltaT_i;				// Set by /I flag.'Delta T' for ions defining a coincidence. 		
	int  minIonsOrder;			// Set by /S flag
	int  maxIonsOrder;			// Set by /W flag
	
	long deltaT_e;				// Set by /E flag.'Delta T' for electrons defining a coincidence.	
	int  minElectronsOrder;		// Set by /L flag
	int  maxElectronsOrder;		// Set by /U flag
	int  correlatedEvent;		// Set by /Z flag
	
	long preambleBytes;			// Bytes to skip at start of file. Set by /S flag.
	char filterStr[256];		// Set by /I= flag.
	int FileOut;
	int AcqTime_s;				//set by /J flag
	int History;				// set by /q
	int fake;					//set by /F
} LoadParams, *LoadParamsPtr;
//-----------------------------------------------------------------------------------------------------------------------------
// GLOBAL VARIABLES 
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Global Variables 

TBuffer	eventBuffer;


//-----------------------------------------------------------------------------------------------------------------------------
// LOCAL VARIABLES
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Local Variables 

// XOP Params
static LoadParams 	lp;
static TListDetector listOfDetector;

// File
static short	a_dump_file_is_defined=0;
static int	end_of_file=0;	// Fin du fichier de lecture
static XOP_FILE_REF ReadOrWriteFileRef;

//-----------------------------------------------------------------------------------------------------------------------------
// LOCAL PROTOTYPES 
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark Local Prototypes 

static void TellFileType(char *fileMessage, char *fileName, int flags, long totalBytes);
static short copy2Buffer(TBuffer *buffer);
//-----
static short lockStacks(TDetector *Detector);
static void unlockStacks(TDetector *Detector);
static short updateIndexStacks(TDetector *Detector, int mode);
static short updateDataStacks(TDetector *Detector, int mode);
static short updateStacks(TListDetector *ListOfDetector, int mode);
//-----
static short getnextEvent(TBuffer *buffer, float *read, float *correct, float *error, float *toolong);
static short checkEvent(TListDetector *ListOfDetector, TBuffer *buffer, int nhits);
static short storeEvent(TListDetector *ListOfDetector);
//-----
static short InitParameters();
static short GetLoadFlags(char symbolicPathName[MAX_OBJ_NAME+1]);
static short GetData( );

//-----------------------------------------------------------------------------------------------------------------------------
//  									  CODE  
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------------------------------------------
/*	TellFileType(fileMessage, fileName, flags, totalBytes)
	Shows type of file being loaded in history.
*/
static void TellFileType(char *fileMessage, char *fileName, int flags, long totalBytes)
{
	char temp[MAX_PATH_LEN + 100];
	
	if ((flags & QUIET) == 0) {
		sprintf(temp, "%s \"%s\" (%ld total bytes)\015", fileMessage, fileName, totalBytes);
		XOPNotice(temp);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------
static short copy2Buffer(TBuffer *buffer)
{
unsigned long nBytesRead;
int result=0;

	
#ifdef CONNECTED_ON_VME
	if(I_vme==0){	// first call or end of the buffer
		result = transfer_a_whole_bunch( (unsigned long)BUFFER_DIMENSION, VME);	if(result) return(result);
	}
	if (I_vme>=nBytesRead) return(-1);	

	*ch = VME[I_vme++];
#else
	result = XOPReadFile(ReadOrWriteFileRef, (long)BUFFER_SIZE, buffer->data, &nBytesRead);	if(result) return(result);
	if(nBytesRead == 0) { //Time out or end of file
			resetBuffer(buffer);
			return -1;
		}
	else {
			buffer->ptrEnd=buffer->data +nBytesRead;
		}			
	return(result);
#endif
}

//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------

//-----------------------------------------------------------------------------------------------------------------------------
static short lockStacks(TDetector *Detector)
{
	int error=0;
	long type,dataOffset[5];
	type=Detector->type;
	
	if (error=MDAccessNumericWaveData(Detector->w_p,kMDWaveAccessMode0,&dataOffset[0])) return error;
	Detector->hState[0] = MoveLockHandle(Detector->w_p);
	if (error=MDAccessNumericWaveData(Detector->w_n,kMDWaveAccessMode0,&dataOffset[1])) return error;
	Detector->hState[1] = MoveLockHandle(Detector->w_n);
	if (type &1) {
		if (error=MDAccessNumericWaveData(Detector->w_t,kMDWaveAccessMode0,&dataOffset[2])) return error;	
		Detector->hState[2] = MoveLockHandle(Detector->w_t);
		}
	if (type&2) {
		if (error=MDAccessNumericWaveData(Detector->w_x,kMDWaveAccessMode0,&dataOffset[3])) return error;
		Detector->hState[3] = MoveLockHandle(Detector->w_x);
		}
	if (type&4) {
		if (error=MDAccessNumericWaveData(Detector->w_y,kMDWaveAccessMode0,&dataOffset[4])) return error;
		Detector->hState[4] = MoveLockHandle(Detector->w_y);	
		}
	return error;
}
//-----------------------------------------------------------------------------------------------------------------------------
static void unlockStacks(TDetector *Detector)
{
	long type;
	type=Detector->type;
	
	HSetState((Handle)Detector->w_p,Detector->hState[0]);	
	WaveHandleModified(Detector->w_p);
	HSetState((Handle)Detector->w_n,Detector->hState[1]);	
	WaveHandleModified(Detector->w_n);
	if (type &1) {
		HSetState((Handle)Detector->w_t,Detector->hState[2]);	
		WaveHandleModified(Detector->w_t);
		}
	if (type&2) {
		HSetState((Handle)Detector->w_x,Detector->hState[3]);	
		WaveHandleModified(Detector->w_x);
		}
	if (type&4) {
		HSetState((Handle)Detector->w_y,Detector->hState[4]);	
		WaveHandleModified(Detector->w_y);	
		}				
}
//-----------------------------------------------------------------------------------------------------------------------------
static short updateIndexStacks(TDetector *Detector, int mode)	
{
	long numRows=0, numColumns=0;
	int error=0,flags =OVERWRITE;
	char	s[100],*suffix;
	waveHndl	w;

	suffix=Detector->suffix;
	switch(mode) {
		case CREATE:
			sprintf(s,"p_%s",suffix); if (error=MakeAWave(s, flags, &Detector->w_p, INCREASE_STACK,1, NT_I32)) return error;
			sprintf(s,"n_%s",suffix); if (error=MakeAWave(s, flags, &Detector->w_n, INCREASE_STACK,1, NT_I32)) return error;
			Detector->PtrIndexStack = 0;
			Detector->EndIndexStack = INCREASE_STACK;
		break;
		
		case ADD:
			sprintf(s,"p_%s",suffix); if (error=CheckWave(s, &Detector->w_p, &numRows, &numColumns)) return error;
			sprintf(s,"n_%s",suffix); if (error=CheckWave(s, &Detector->w_n, &numRows, &numColumns)) return error;
			Detector->PtrIndexStack = numRows;
			numRows += INCREASE_STACK;
			sprintf(s,"Redimension/N=%ld p_%s,n_%s",numRows,Detector->suffix,Detector->suffix);	
			if (error=XOPSilentCommand(s)) return error;
			Detector->EndIndexStack = numRows ;
		break;		

		case INCREASE_SIZE:	// On arrive ici avec PtrIndexStack = EndIndexStack c.a.d. plus de place dans les tableaux
			unlockStacks(Detector);			
			sprintf(s,"p_%s",suffix); if (error=CheckWave(s, &w, &numRows, &numColumns)) return error;
			sprintf(s,"n_%s",suffix); if (error=CheckWave(s, &w, &numRows, &numColumns)) return error;
			if(numRows != Detector->PtrIndexStack) {   // Error ???
					sprintf(s, "numRows= %ld,PtrIndexStack= %ld",numRows,Detector->PtrIndexStack);	
					XOPNotice(s);
				}			
			numRows += INCREASE_STACK;
			sprintf(s,"Redimension/N=%ld p_%s,n_%s",numRows,suffix,suffix);	
			if (error=XOPSilentCommand(s)) return error;
			Detector->EndIndexStack = numRows ;
			if (error=lockStacks(Detector)) return error;
		break;
				
		case END_OF_ACQUISITION:
			if (Detector->PtrIndexStack <2) 	
				sprintf(s,"Redimension/N=%ld p_%s,n_%s",2,suffix,suffix);
			else 				
				sprintf(s,"Redimension/N=%ld p_%s,n_%s",Detector->PtrIndexStack,suffix,suffix);
			if (error=XOPSilentCommand(s)) return error;
			Detector->EndIndexStack = Detector->PtrIndexStack ;
		break;
	}
	return 0;
}

//-----------------------------------------------------------------------------------------------------------------------------
static short updateDataStacks(TDetector *Detector, int mode)	
{
	long numRows=0, numColumns=0, type;
	int error=0,flags =OVERWRITE;
	char s0[100],s1[100],*suffix;
	waveHndl	w;
	
	suffix=Detector->suffix;
	type=Detector->type;
	switch(mode) {
		case CREATE:
			if (type&1) {
				sprintf(s1,"t_%s",suffix);
				if (error=MakeAWave(s1, flags, &Detector->w_t, INCREASE_STACK, 1, NT_FP32)) return error;
				}
			if (type&2) {
				sprintf(s1,"x_%s",suffix);
				if (error=MakeAWave(s1, flags, &Detector->w_x, INCREASE_STACK, 1,NT_FP32)) return error;
				}
			if (type&4) {
				sprintf(s1,"y_%s",suffix);
				if (error=MakeAWave(s1, flags, &Detector->w_y, INCREASE_STACK, 1,NT_FP32)) return error;
				}
			Detector->PtrDataStack = 0;
			Detector->EndDataStack = INCREASE_STACK;
		break;

		case ADD:
			s0[0]=0;
			if (type&1) {
				sprintf(s1,"t_%s",suffix); 
				if (error=CheckWave(s1, &Detector->w_t, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			if (type&2) {
				sprintf(s1,"x_%s",suffix); 
				if (error=CheckWave(s1, &Detector->w_x, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			if (type&4) {	
				sprintf(s1,"y_%s",suffix); 
				if (error=CheckWave(s1, &Detector->w_y, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			s0[strlen(s0)-1]=0;
			Detector->PtrDataStack = numRows;
			numRows += INCREASE_STACK;
			sprintf(s1,"Redimension/N=%ld %s",numRows, s0);
			if (error=XOPSilentCommand(s1)) return error;
			Detector->EndDataStack = numRows;
		break;
		
		case INCREASE_SIZE:					
			unlockStacks(Detector);
			s0[0]=0;
			if (type&1) {
				sprintf(s1,"t_%s",suffix); 
				if (error=CheckWave(s1, &w, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			if (type&2) {
				sprintf(s1,"x_%s",suffix); 
				if (error=CheckWave(s1, &w, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			if (type&4) {	
				sprintf(s1,"y_%s",suffix); 
				if (error=CheckWave(s1, &w, &numRows, &numColumns)) return error;
				sprintf(s0,"%s %s,",s0, s1);
				}
			s0[strlen(s0)-1]=0;
			if(Detector->PtrDataStack+Detector->nTimeGroups!=numRows) {
				sprintf(s1," Problems in updateDataStacks -> Ptr=%i, NumRows=%i, Event multiplicity=%i \015", 
						Detector->PtrDataStack,numRows,Detector->nTimeGroups);
				XOPNotice(s1);
				}
			numRows += INCREASE_STACK;
			sprintf(s1,"Redimension/N=%ld %s",numRows, s0);
			if (error=XOPSilentCommand(s1)) return error;
			Detector->EndDataStack = numRows;
			if (error=lockStacks(Detector)) return error;
		break;
		
		case END_OF_ACQUISITION:
			if (Detector->EndDataStack<2) 
				{sprintf(s0,"Redimension/N=%ld",2);}
			else 
				{sprintf(s0,"Redimension/N=%ld",Detector->PtrDataStack);}
				
			if (type&1) { sprintf(s0,"%s t_%s,",s0, suffix);}
			if (type&2) { sprintf(s0,"%s x_%s,",s0, suffix);}
			if (type&4) { sprintf(s0,"%s y_%s,",s0, suffix);}
			s0[strlen(s0)-1]=0;
			if (error=XOPSilentCommand(s0)) return error;
			Detector->EndDataStack = Detector->PtrDataStack;
		break;
	}
	return 0;
}
//-----------------------------------------------------------------------------------------------------------------------------
static short updateStacks(TListDetector *ListOfDetector, int mode)
{	
	TDetector *Detector;
	int i,error=0;
	
	// A revoir... 
	for (i=0;i<ListOfDetector->nbDetector;i++) {
		Detector=&ListOfDetector->Detector[i];	
		switch (mode) {
			case CREATE :
					if(error=updateIndexStacks(Detector, mode)) return error;
					if(error=updateDataStacks(Detector, mode)) return error;
					if(error=lockStacks(Detector)) return error;
				break;
			case ADD :
					if(error=updateIndexStacks(Detector, mode)) return error;
					if(error=updateDataStacks(Detector, mode)) return error;	
					if(error=lockStacks(Detector)) return error;
				break;
			case END_OF_ACQUISITION :
					unlockStacks(Detector);
					if(error=updateIndexStacks(Detector, mode)) return error;
					if(error=updateDataStacks(Detector, mode)) return error;
				break;
		}
		if (error=updateDetector(Detector, mode)) return error;												
	}		
	return 0;	
}
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------
static short getnextEvent(TBuffer *buffer, float *read, float *correct, float *error, float *toolong)
{ 	
	unsigned char ok=0, fastforward=0;
	long nhits;
	
	do {
		if (readBuffer(buffer,8,&nhits)) return -1;  // Read "buffer" in "8" bits mode and return the "nhits"
		if (nhits!=0xFF) {							 // Is the nb of hits a end marker ?
			if (!fastforward) {						 // Is not the fastforward mode to the next event activated ?
				*(read)+=1;							 				// yes, then consider this event
				if (ismarkerBuffer(0xFF, buffer, nhits, buffer->type)) { // Is a end marker ? 
					if (nhits<N_HITS_MAX) ok=1;						// Is the size not too long ?
					else {	// Too long -> suspicious ??? 
						ok=0;
						*(toolong)+=1;
					}
				}
				else {		// Data not coherent if no marker 0xFF after nhits*(16+8)bits
					fastforward=1;	// activate the fast foward mode
					*(error)+=1; 	
				}
			}
		}				 
		else fastforward=0; // stop the fast forward mode
	} while (!ok);
	*(correct)+=1;
	return nhits;
}

//-----------------------------------------------------------------------------------------------------------------------------
static short checkEvent(TListDetector *ListOfDetector, TBuffer *buffer, int nhits)
{ 
	TDetector *Detector;
	long wordbuffer, tdcTime,tdcChannel;
	long wordsize = ListOfDetector->wordsizeEvent;
	long maskTime = ListOfDetector->maskTime;
	long sigbitTime = ListOfDetector->sigbitTime;
	long maskChannel = ListOfDetector->maskChannel; 
	long sigbitChannel = ListOfDetector->sigbitChannel;
	long maskDetector = ListOfDetector->maskDetector;
	long sigbitDetector = ListOfDetector->sigbitDetector;
	int i, mode, numDetector; 
	short error=0;
	
	mode=ListOfDetector->modeDetector;	
	
	// Initialize "Event" structure depending on the detector type
	for (numDetector=0;numDetector<ListOfDetector->nbDetector;numDetector++) 
		newEvent(&ListOfDetector->Detector[numDetector]);
	
	// 
		
	// Determine the reference time 
		
	for(i=0;i<nhits;i++){
		if (readBuffer(buffer, wordsize, &wordbuffer)) return -1;
		numDetector=(wordbuffer&maskDetector)>>sigbitDetector;
		Detector=&ListOfDetector->Detector[numDetector];
		tdcTime=(wordbuffer&maskTime)>>sigbitTime;				
		tdcChannel=(wordbuffer&maskChannel)>>sigbitChannel;		
		if (error=addEvent(Detector,tdcTime, tdcChannel)) return -1;	
	}		
	
	if (error=correctEvent(ListOfDetector)) return -1;		
	
	return 0;
}

//-----------------------------------------------------------------------------------------------------------------------------
static short storeEvent(TListDetector *ListOfDetector)
{
	TDetector *Detector;
	long nTimeGroups,type;
	float *xMoyen, *yMoyen,*MeanTimeofGroup;
	long *PtrIndexStack, *EndIndexStack;
	long *PtrDataStack, *EndDataStack;	
	long *PtrP,*PtrN;
	float *PtrT,*PtrY,*PtrX;
	
	int i,j,error=0;
		
#ifdef GARDER_LES_EVENEMENTS	
	for (i=0;i<ListOfDetector->nbDetector;i++) {
		Detector=&ListOfDetector->Detector[i];
		// Event Info
		type= Detector-> type;
		nTimeGroups = Detector->nTimeGroups;		
		xMoyen = Detector->xMoyen;
		yMoyen = Detector->yMoyen;
		MeanTimeofGroup = Detector->MeanTimeofGroup; 
		
		// Stack info 
		PtrIndexStack = &Detector->PtrIndexStack;
		EndIndexStack = &Detector->EndIndexStack;
		PtrDataStack = &Detector->PtrDataStack;
		EndDataStack = &Detector->EndDataStack;
		
		// Check that the stacks are big enough to store master index and data
		if (*PtrIndexStack >= *EndIndexStack) 			 error=updateIndexStacks(Detector, INCREASE_SIZE);
		if (error) return error;
		if (*PtrDataStack+nTimeGroups >= *EndDataStack)  error=updateDataStacks(Detector, INCREASE_SIZE);
		if (error) return error;	
		
		// 	First, save the master index = reference to the data in a stack, multiplicity of the event
		PtrP=WaveData(Detector->w_p);
		PtrN=WaveData(Detector->w_n);
		*(PtrP+*PtrIndexStack)=*PtrDataStack;
		*(PtrN+*PtrIndexStack)=nTimeGroups;
		
		//	Then, save the data in a stack = time, x, y...
		PtrT=WaveData(Detector->w_t);
		PtrX=WaveData(Detector->w_x);
		PtrY=WaveData(Detector->w_y);			
		for (j=0;j<nTimeGroups;j++){
			if (type&1) { *(PtrT+*PtrDataStack) = MeanTimeofGroup[j]; }				
			if (type&2) { *(PtrX+*PtrDataStack) = xMoyen[j]; }
			if (type&4) { *(PtrY+*PtrDataStack) = yMoyen[j]; }			
			*PtrDataStack+=1; // increase stack pointer;	
		}
	*PtrIndexStack+=1; // Increment the pile for the next event
	}
#endif
	return error;
}

//-----------------------------------------------------------------------------------------------------------------------------
short loopEvent()					
{
	float 	nbRead=0, nbCorrect=0, nbError=0, nbToolong=0;
	long	StartTime;
	int 	nhits, error;
	char	s[255];

	WatchCursor();
	StartTime =TickCount();

init:	
	error=0;
	resetBuffer(&eventBuffer);
	if (copy2Buffer(&eventBuffer)) goto exit_Read_Events; // There is no more data to copy into the buffer
	
loop:		
	nhits=getnextEvent(&eventBuffer, &nbRead, &nbCorrect, &nbError, &nbToolong); 
	if (nhits==-1) goto init; // Buffer is empty read again from file or VME
				
	// Check Event
	error=checkEvent(&listOfDetector, &eventBuffer, nhits); 
	if (error) goto loop;
	if (!lp.calibrate) {
		error=analyzeEvent(&listOfDetector); 
		if (error) goto loop;
		error=storeEvent(&listOfDetector);
		if (error) goto exit_Read_Events;
	}
	goto loop;

exit_Read_Events:
	ArrowCursor();
	
	if (!lp.calibrate) { 
		sprintf(s,"Nb of events :%.0f - Amount of correct : %.3f %%-> Error in event format: %.3f %%- Too many hits per events: %.3f %% \015", 
		nbRead, 100*nbCorrect/nbRead, 100*nbError/nbRead, 100*nbToolong/nbRead );
		XOPNotice(s);
		sprintf(s,"ReadData : %ld Ticks\015",(long)(TickCount()-StartTime));XOPNotice(s);	
	}
	return(error);
}
//-----------------------------------------------------------------------------------------------------------------------------
#pragma mark ----------------------------
//-----------------------------------------------------------------------------------------------------------------------------
static short InitParameters()
{
//	MemClear(lp, sizeof(LoadParams));
//	lp.flags = 0;
//	lp.Acquire = 0;								// printout for debug treasons only !

	lp.FileOut =1;					//  /T inhibits output file

	lp.acquire = 0;
	lp.add = 0 ;	
	lp.mode = 0;
	lp.AcqTime_s=15;
	lp.calibrate=0;
	
	lp.deltaT_e = 3; 					// 1.5 ns 	
	lp. minElectronsOrder=1;
	lp.maxElectronsOrder = 1;
	lp.deltaT_i = 6; 					// 3 ns 
	lp.minIonsOrder=1;
	lp.maxIonsOrder = 1;
	lp.correlatedEvent=0;
	
	lp.preambleBytes = 0;
							// infinite data collection loop
	lp.fake=0;							// do not read the fake data on vme
	strcpy(lp.filterStr,"CIEL");
	end_of_file = 0;
	lp.History=0;
	return 0;
}

//-----------------------------------------------------------------------------------------------------------------------------
/*	GetLoadFlags(lpp, symbolicPathName)
	GetLoadFlags parses any flags in the command line.
	It sets the fields in lpp->flags using one bit for each flag as defined in ReadData.h.
	If there is a /P=path flag it returns the symbolic path name via symbolicPathName.
	Other parameters are returned via the LoadParams structure.
	
	It returns 0 if OK or an error code otherwise.
*/
static short GetLoadFlags(char symbolicPathName[MAX_OBJ_NAME+1])
{
DOUBLE 	d1;
int 		result, flag,int1;
	
	lp.flags = 0;
	*symbolicPathName = 0;						// This means no symbolic path was specified.

	while (flag = GetFlag("MACPOEINUSLFWVTJ")) {
		if (flag < 0)
			return BADFLG;
			
		switch(flag) {
			case 'M':							// Predifined acquisiton mode
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				int1 = d1;
				lp.mode = int1;
				break;
				
			case 'C':							// /A[=v] sets Add.
				lp.calibrate = 1;
				break;
												
			case 'A':							// Acquisition from VME
				lp.add=1;
				break;

			case 'J':							// /J= acquisition time
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				int1 = d1;
				lp.AcqTime_s = int1;
				break;
								
			case 'P':							// /P means symbolic path coming.
				if (GetSymb() != '=')	return NOEQUALS;
				if (result = GetAString(symbolicPathName))	return result;
				if (*symbolicPathName == 0)	break;						// Treat /P=$"" as a NOP.
				//if (result = GetPathInfo2(symbolicPathName, fileDirPath))	return result;				// Path does not exist.
				lp.flags |= PATH;
				break;
				
			case 'O':							// /O means overwrite waves.
				if (result = GetTrueOrFalseFlag(OVERWRITE, &lp.flags))	return result;
				break;
				
				
			case 'E':							// /E means 'Delta T' for electrons defining a coincidence.
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.deltaT_e =(long)d1;
				break;
				
			case 'L':							// /L=s sets number of min electrons in an event
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.minElectronsOrder = d1;
				break;
				
			case 'U':							// /U=u sets number max electrons in an event.
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.maxElectronsOrder = d1;
				break;
				
			case 'I':								// /I means 'Delta T' for ions defining a coincidence.
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.deltaT_i =(long)d1;
				break;
				
			case 'S':							// /S=s sets number of  min ions in an event
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.minIonsOrder = d1;
				break;
				
			case 'W':							// /W=w sets number of  max ions in an event
				if (GetSymb() != '=')			return NOEQUALS;
				if (result = GetFlagNum(&d1))	return result;
				lp.maxIonsOrder = d1;
				if ((lp.maxIonsOrder < 0)	|| (lp.maxIonsOrder > 4))		return BAD_NUM_WAVES;
				break;		
							
			case 'F':							// /F reads the test data on VME
				lp.fake = 1;
				break;
				
			case 'T':							// /T= no output file
				lp.FileOut = 0;
				break;
				
		}
	}
	// Check if the parameter are coherent !!!
	//	if((lp.maxIonsOrder<1)||(lp.minIonsOrder>1)||(lp.maxIonsOrder==0)) goto WhatNextWithIon;
	if (lp.mode) createDetectorList(&listOfDetector,lp.mode);
	return 0;
}
//-----------------------------------------------------------------------------------------------------------------------------
/*	GetData(ReadOrWriteFileRef, lp, caPtr, numColumns, waveNamesHandle)

	Loads data into waves.
	lp is a pointer to a structure containing parameters to the Start operation.
*/
static short GetData( )
{
	short error=0;
	
	// Get information from Igor variables (in case of user changes in the control panel)
	if (error = getigorDetectorList(&listOfDetector)) {
		createDetectorList(&listOfDetector,2);
	 	return error;
		}
		
	// Initialize the system to get data
	createBuffer(&eventBuffer,listOfDetector.wordsizeEvent);
	if (lp.add)
		error=updateStacks(&listOfDetector,ADD);
	else
		error=updateStacks(&listOfDetector,CREATE);
	

	// Starts acquistion loop
#ifndef CONNECTED_ON_VME
	if (!error) error = loopEvent();
	error=updateStacks(&listOfDetector, END_OF_ACQUISITION);
	if (XOPAtEndOfFile(ReadOrWriteFileRef))		error = 0;
	else error = -1;
#else

#endif		
	return error;
}


//-----------------------------------------------------------------------------------------------------------------------------
/*	Start()
	Start is called when XOP is called from the host application with message = CMD.
	Start returns 0 if everything went alright or an error code if not.
*/
short Start(void)
{
char 		symbolicPathName[MAX_OBJ_NAME+1];		// Name of the symbolic path or "".
char 		symbolicPathPath[MAX_PATH_LEN+1];		// Full path to the folder that the symbolic path refers to. This is a native path with trailing colon (Mac) or backslash (Win).
char 		filePath[MAX_PATH_LEN+1];				// May be full path, partial path, or just a file name.
char 		fileDirPath[MAX_PATH_LEN+1];
char 		fileName[MAX_FILENAME_LEN+1], s[255];
int 			err;
long			updateSave;
unsigned long	fileBytes=0;
int*			dummy=0;
	
	// Initialize parameters to default value
	InitParameters(&lp);
	// Get parameters from User command line
	if (err = GetLoadFlags(symbolicPathName))		return err;
	if (lp.mode) return 0;
	//	Prepare input/output : we are going to define the variable FilePath for the file 
	// which will contain the VME dump, OR contains the dump that we are going to read
	a_dump_file_is_defined =0;	
	
#ifdef CONNECTED_ON_VME
//	if (lp.add !=0) {
		if(lp.FileOut !=0) {
// 		File to write the VME DUMP
			err = XOPSaveFileDialog("File to save the VME Dump","",dummy,"","",filePath);
			if(err==0){
					err = XOPCreateFile(filePath,1,'MLAV','CIEL');	if(err) return err;	
					XOPNotice("A VME dump will be written in the file  ");
					XOPNotice(filePath);	XOPNotice("'\015");		//  (14 mars 2000)
					a_dump_file_is_defined =1;
					// 	Open file and get ReadOrWriteFileRef
					if (err = XOPOpenFile(filePath, 1, &ReadOrWriteFileRef))	return err;
					if (err = XOPSetFilePosition(ReadOrWriteFileRef,0,-1))		return err;
			}
		}
//	}	// end of if (lp.add !=0)
#else
	// Check whether a filename is already given

	*filePath = 0;
	if (IsStringExpression(1)) {	// Is the next item is a string?
			if (err = GetAString(filePath))	return err;		// Get file name or path and name.
			if (err = CheckTerm())			return err;		// Make sure no extraneous garbage in cmd.
			if (err = GetNativePath(filePath, filePath))	return err;	 	// Convert Mac path to Win path or vice versa, if needed.
		}
				
	else { // Need dialog to get file name.
			if (err = XOPOpenFileDialog("Looking for a CIEL file", lp.filterStr, NULL, symbolicPathPath, filePath)) return err;
			// 	Open file and get ReadOrWriteFileRef
			}		
	if (err = XOPOpenFile(filePath, 0, &ReadOrWriteFileRef)) return err;
	if (err = XOPSetFilePosition(ReadOrWriteFileRef,0,-1))	return err;	
	
	a_dump_file_is_defined =1;
#endif
	if(a_dump_file_is_defined==1) {
			if (err = GetDirectoryAndFileNameFromFullPath(filePath, fileDirPath, fileName))	return err;
			// Initialize file loader global output variables.
			if (err = SetFileLoaderOutputVariables(filePath, 0, ""))	return err;
		}

	WatchCursor();
	PauseUpdate(&updateSave);						// Don't control windows during this.	
	if (!lp.calibrate) {
		strcpy(s,"print \" Lecture le\",Date(),\"\",time()");XOPSilentCommand(s);
	}
	
#ifdef CONNECTED_ON_VME
	Init_8255();					// This stops the experiment
	if(lp.fake!=1)		Clean_The_VME_Buffer();		// kick away all the remaining stuff in the VME		
	RESUME_ACQ;
	TargetTicks = TickCount() + lp.AcqTime_s*60;
#else
	// Find the total number of bytes in the file.
	if (err = XOPNumberOfBytesInFile(ReadOrWriteFileRef, &fileBytes))	return err;
	strcpy(s, "Load from");
	if (!lp.calibrate) {
		TellFileType(s, fileName, lp.flags, fileBytes);	// Figure out number of waves and points/wave.	
		}
	fileBytes -= lp.preambleBytes;
#endif
	// Load data.
	err = GetData(&lp);

	ResumeUpdate(&updateSave);						// Go back to previous update state.
	
#ifdef CONNECTED_ON_VME
	//	if the acquisition has been stopped (command "pause")
 	if (acquisState == STOPPED) XOPCloseFile(ReadOrWriteFileRef);
#else
	XOPCloseFile(ReadOrWriteFileRef);
#endif	
	return err;
}
//-----------------------------------------------------------------------------------------------------------------------------
void Stop()
{
#ifdef CONNECTED_ON_VME
//	char s[255];
		SUSPEND_ACQ;
		loopEvent();
		if(a_dump_file_is_defined==1) XOPCloseFile(ReadOrWriteFileRef);
		//sprintf(s,"Total of Bytes saved to Disk = %ld KBytes", TotalByteFromVME/1024);
		//XOPNotice(s);
#endif
}
//-----------------------------------------------------------------------------------------------------------------------------
// All structures are 2-byte-aligned.
#if TARGET_CPU_PPC
	#pragma options align=reset
#endif
#ifdef _WINDOWS_
	#pragma pack()
#endif
